﻿"""
Implementation of a walking transport.

Note that the walking transport uses finalize in order to ensure
that the movement for each frame is at the same rate even when moving diagonally.

http://en.wikipedia.org/wiki/Preferred_walking_speed
- 1.4 m/s

http://en.wikipedia.org/wiki/Running_speed
- 10.44 m/s

"""

import vizmat

from transportation import AccelerationTransport


class Walking(AccelerationTransport):
	"""A simple transport for walking around a scene."""
	
	def __init__(self,
					height=0,
					acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
					maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
					rotationAcceleration=90.0, # in degrees per second per second
					maxRotationSpeed=120.0, # in degrees per second
					autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
					dragCoef=0.0001,
					rotationAutoBreakingDragCoef=0.2, # determines how quickly the walking transport will stop 
					rotationDragCoef=0.0001, # normal drag coef
					**kwargs):
		
		super(Walking, self).__init__(acceleration=acceleration,
								maxSpeed=maxSpeed,
								rotationAcceleration=rotationAcceleration,
								maxRotationSpeed=maxRotationSpeed,
								autoBreakingDragCoef=autoBreakingDragCoef,
								dragCoef=dragCoef,
								rotationAutoBreakingDragCoef=rotationAutoBreakingDragCoef,
								rotationDragCoef=rotationDragCoef,
								**kwargs)
		self.setPosition(0, height, 0)
		
		self.setUpdateFunction(lambda transport: None)
	
	def moveForward(self, mag=1):
		"""Moves forward"""
		self._Ap[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveBackward(self, mag=1):
		"""Moves backward"""
		self._Ap[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveLeft(self, mag=1):
		"""Moves left"""
		self._Ap[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveRight(self, mag=1):
		"""Moves right"""
		self._Ap[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveUp(self, mag=1):
		"""Moves up"""
		self._Ap[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveDown(self, mag=1):
		"""Moves down"""
		self._Ap[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def lookDown(self, mag=1):
		"""Looks down"""
		self._Ar[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def lookUp(self, mag=1):
		"""Looks up"""
		self._Ar[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def turnLeft(self, mag=1):
		"""Turns left"""
		self._Ar[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def turnRight(self, mag=1):
		"""Turns right"""
		self._Ar[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def getAccelerationRotationMatrix(self):
		"""Returns the acceleration rotation matrix, i.e. the matrix used to
		determine reference frame for acceleration.
		
		@return vizmat.Transform()
		"""
		rotMat = vizmat.Transform()
		rotMat.setEuler([self.getEuler()[0], 0, 0])
		return rotMat


if __name__ == "__main__":
	import viz
	import vizact
	viz.go()
	
	# load a model
	piazza = viz.add('piazza.osgb')
	
	transportRepresentation = viz.add("beachball.osgb")
	walking = Walking(height=1.82,
					acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
					maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
					rotationAcceleration=90.0, # in degrees per second per second
					maxRotationSpeed=120.0, # in degrees per second
					autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
					dragCoef=0.0001,
					rotationAutoBreakingDragCoef=0.2, # determines how quickly the walking transport will stop 
					rotationDragCoef=0.0001, # normal drag coef
					autoBreakingTimeout=0.2, # how long before auto breaking is enabled * see comment
					rotationAutoBreakingTimeout=0.2, # how long before rotational auto breaking is enabled * see comment
					)
	# Note regarding autoBreakingTimeouts:
	# When using a transport, the auto breaking function acts as if the user has
	# applied pressure to a break peddle. Auto breaking occurs whenever the user 
	# stops applying acceleration in a particular direction. So if you remove your
	# finger from the forward key, the auto breaking coefficient in the forward
	# direction is automatically applied. If you'd like to coast to a stop, you
	# can lower or remove (set to 0) the auto breaking coefficient.
	# 
	# When using an update function with a transport, the state of they buttons/
	# keys are constantly sampled. If the data is irregular, it may be best to
	# set an autoBreakingTimeout, which is a minimum time to wait for input signal
	# before auto breaking is used.
	#
	# If you're not setting an update function, but manually calling the moveForward,
	# moveBackward, etc events, then it's generally a good idea to apply some timeout
	# as events triggered by keyboards and other inputs tend to update less than
	# once per frame.
	
	vizact.onkeydown(viz.KEY_UP, walking.moveForward)
	vizact.onkeydown(viz.KEY_DOWN, walking.moveBackward)
	vizact.onkeydown(viz.KEY_LEFT, walking.moveLeft)
	vizact.onkeydown(viz.KEY_RIGHT, walking.moveRight)
	
	vizact.onkeydown('w', walking.lookUp)
	vizact.onkeydown('s', walking.lookDown)
	vizact.onkeydown('a', walking.turnLeft)
	vizact.onkeydown('d', walking.turnRight)
	vizact.onkeydown('z', walking.moveDown)
	vizact.onkeydown('x', walking.moveUp)
	
	base = viz.addGroup()
	box = viz.add("arrow.wrl")
	box.setPosition(0, 0, 1.5)
	box.setScale([0.15]*3)
	box.setParent(base)
	
	vizact.onkeydown('5', walking.setNode, base)
	
	group = viz.addGroup()
	group.setParent(walking)
	group.setEuler([0, 0, 0])
	link = viz.link(group, viz.MainView)
	link.setSrcFlag(viz.ABS_GLOBAL)
#	link = viz.link(walking, viz.MainView)


